/*
Program for receiving and decoding signals from 433.92 MHz AM ASK radio receiver module.
Decodes the NEXA and NEXAL protocol. Simple description of the protocol I've found so far:

NEXA:
First seven pulses in the intervals described in patMinN and patMaxN, then each bit in the data is approximately eighter :
400 900 400 900 - meaning 0 or
900 400 400 900 - meaning 1. The program just looks if the first pulse is > 600 (iolevel)  
bit 0-3 => house number
bit 4-7 => unit number
bit 11  => 0=off, 1=on

NEXAL (NEXA self learning):
First seven pulses in the intervals described in patMinNL and patMaxNL, then each bit in the data is approxiately eighter:
320 200 320 1200 - meaning 0 or
320 1200 320 200 - meaning 1. The program just looks if the second pulse is > 600 (iolevel)
bit 0-25  => house number
bit 26    => is this a group call (1) or not (0) (turn on or off everything)
bit 27    => 0=off, 1=on
bit 28-31 => unit number

Author: Johan Ekblad <jka@ekblad.org> 9 sept 2010 
 
Based on a program by Johan Adler http://www.google.com/profiles/johan.adler

License: GNU/GPL v3  
*/

#include <avr/interrupt.h>
#include "rfrx.h"

#define output Serial
#define outBps 9600

#define UNDEF 0
#define NEXA  1
#define NEXAL 2

unsigned int  patMinN[7]={5000,1000,100,70,500,50,20};
unsigned int  patMaxN[7]={15000,2000,400,200,1000,200,170};

unsigned int  patMinNL[7]={9000,1000,150,70,1800,20,120};
unsigned int  patMaxNL[7]={13000,2000,250,200,3000,100,250};

unsigned int  iolevel=600;
  
int           patSize=7;
int           type=UNDEF;        //0=undef, 1=nexa, 2=nexaL
int           matchPos=0;
int           readData=0;
int           bitPos=0;
int           skipCount=0;
unsigned long house=0;
unsigned int  group=0;
unsigned int  unit=0;
unsigned int  mode=0;
unsigned long lastSent;
unsigned long loopc = 0;
  
void writeProtocolData()
{
    unsigned long now=millis();
    if (now-lastSent > 1000) // Only send again if more than one second since last time
    {
        if (type == NEXA)
        {
            output.println("Protocol:NEXA");
        }  
        else if (type == NEXAL)
        {
            output.println("Protocol:NEXAL");
        }
        output.print("House:");
        output.print(house,DEC);
        output.println();
        output.print("Unit:");
        output.print(unit,DEC);
        output.println();
        output.print("Mode:");
        output.print(mode,DEC);
        output.println();
        
        if (type == NEXAL)
        {
            output.print("Group:");
            output.print(group,DEC);
            output.println();
        }
        
        output.println();
        
        lastSent=now;
    }
    
    // Zero out everything
    bitPos=0;
    readData=0;
    matchPos=0;   
    skipCount=0; 
    house=0;
    group=0;
    unit=0;
    mode=0; 
    type=UNDEF;     
  
}

void parseNexaL(unsigned int bit)
{


    if (bitPos >=1 && bitPos <= 26)
    {
        house=house*2+bit;
    }
    else if (bitPos == 27)
    {
        group=bit;
    }
    else if (bitPos == 28)
    {
        mode = bit;
    }
    else if (bitPos >=29 && bitPos <= 32)
    {
        unit=unit*2+bit;
    }
    if (bitPos == 32) // End of protocol
    {
        writeProtocolData();
    }
}

void parseNexa(unsigned int bit)
{
    if (bitPos >=1 && bitPos <= 4)
    {
        house=house*2+bit;
    }
    else if (bitPos >=5 && bitPos <= 8)
    {
        unit=unit*2+bit;
    }
    if (bitPos == 12)
    {
        mode=bit;
        writeProtocolData();
    }
}

void setup(void) {
  output.begin(outBps);
  icp_init();
  lastSent=0;
  unsigned long loopc=0;
}

void loop(void) 
{
    if (icpAvail())
    {
        rxDataStruct s = icpRead();
        unsigned int data=s.dur;
        
        loopc++;
/*
        Serial.print(data);
        Serial.print(" ");
        if (loopc % 20 == 0)
        {
           Serial.println();
        }
*/        
        if (skipCount)
        {
            skipCount--;
          
            if (skipCount == 2 && type == NEXAL) // NexaL case 320 200 320 1200 (0) - 320 1200 320 200 (1)
            {
                int bit=0;
                if (data > iolevel)
                {
                    bit=1;
                }       
                bitPos++;
                parseNexaL(bit);      
            }   
        }
        else if (readData)  
        {
            int bit=0;
            skipCount=3;
           
            if (type == NEXA)  // Nexa case 400 900 400 900 (0) - 900 400 400 900 (1)
            {           
                if (data > iolevel)
                {
                    bit=1;
                }
                bitPos++;
                parseNexa(bit);
            }           
        }
        else if ((type ==UNDEF || type == NEXA) && data >= patMinN[matchPos] && data <= patMaxN[matchPos])
        {
           type=NEXA;         
           matchPos++;
//           Serial.print(matchPos,DEC);
           if (matchPos == patSize)
           {
               readData=1;  
           }
        }                       
        else if ((type == UNDEF || type == NEXA || type == NEXAL) && ((data >= patMinNL[matchPos] && data <= patMaxNL[matchPos]) || (matchPos==1 && data < 100)))
        {                               //  ^--NEXAL matches first 4 pulses of NEXA protocol, continue to match NEXAL   
           type=NEXAL;
           matchPos++;
/*
           Serial.print("m=");
           Serial.print(matchPos,DEC);
           Serial.print(" ");

*/
           if (matchPos == patSize)
           {
               readData=1;  
           }
        } 
        else
        {
           matchPos=0;
           type=UNDEF;
        }
    }
}

